class XRayWebSocketClient {
    constructor(serverConfig) {
        this.server = serverConfig;
        this.ws = null;
        this.isConnected = false;
        this.onConnected = null;
        this.onError = null;
        this.reconnectAttempts = 0;
        this.maxReconnectAttempts = 3;
    }

    async connect() {
        return new Promise((resolve, reject) => {
            try {
                const wsUrl = `wss://${this.server.host}:${this.server.port}${this.server.path}`;
                
                this.ws = new WebSocket(wsUrl);
                
                const timeout = setTimeout(() => {
                    reject(new Error('Connection timeout'));
                }, 10000);

                this.ws.onopen = () => {
                    clearTimeout(timeout);
                    this.isConnected = true;
                    this.reconnectAttempts = 0;
                    if (this.onConnected) this.onConnected();
                    resolve();
                };
                
                this.ws.onerror = (error) => {
                    clearTimeout(timeout);
                    if (this.onError) this.onError(error);
                    reject(error);
                };
                
                this.ws.onclose = (event) => {
                    this.isConnected = false;
                    
                    if (this.reconnectAttempts < this.maxReconnectAttempts) {
                        this.reconnectAttempts++;
                        setTimeout(() => this.connect(), 2000);
                    }
                };

            } catch (error) {
                reject(error);
            }
        });
    }

    disconnect() {
        this.reconnectAttempts = this.maxReconnectAttempts;
        
        if (this.ws) {
            this.ws.close();
            this.ws = null;
        }

        this.isConnected = false;
    }
}

class VPNManager {
    constructor() {
        this.backendUrl = 'http://91.84.117.49:8080';
        this.servers = [];
        this.serverStats = new Map();
        this.currentServer = null;
        this.xrayClient = null;
        this.isConnected = false;
        
        this.initialize();
    }

    async initialize() {
        try {
            const status = await this.loadStatus();
            if (status && status.isConnected && status.currentServer) {
                this.isConnected = true;
                this.currentServer = status.currentServer;
                await this.autoReconnect();
            }
            
            await this.loadServers();
        } catch (error) {}
    }

    async autoReconnect() {
        try {
            if (this.isConnected && this.currentServer) {
                this.xrayClient = new XRayWebSocketClient(this.currentServer);
                
                this.xrayClient.onConnected = () => {};

                this.xrayClient.onError = (error) => {
                    this.isConnected = false;
                    this.saveStatus();
                };

                await this.xrayClient.connect();
                await this.setBrowserProxy();
            }
        } catch (error) {
            this.isConnected = false;
            await this.saveStatus();
        }
    }

    async loadServers() {
        try {
            const response = await fetch(`${this.backendUrl}/api/servers`);
            
            if (!response.ok) {
                throw new Error(`HTTP ${response.status}: ${response.statusText}`);
            }
            
            const data = await response.json();
            this.servers = data;
            
            this.servers.forEach(server => {
                if (!this.serverStats.has(server.id)) {
                    this.serverStats.set(server.id, {
                        connectionCount: 0,
                        lastUsed: 0,
                        responseTime: null
                    });
                }
            });
            
            return this.servers;
        } catch (error) {
            this.servers = [
                { 
                    id: 1, 
                    host: '91.84.117.49', 
                    port: 443, 
                    country: 'nl', 
                    path: '/xray',
                    ip: '91.84.117.49'
                },
                { 
                    id: 2, 
                    host: '144.124.227.90', 
                    port: 443, 
                    country: 'de', 
                    path: '/xray',
                    ip: '144.124.227.90'
                }
            ];
            return this.servers;
        }
    }

    selectOptimalServer() {
        if (this.servers.length === 0) return null;
        
        let bestServer = this.servers[0];
        let minConnections = this.serverStats.get(bestServer.id)?.connectionCount || 0;
        
        for (const server of this.servers) {
            const stats = this.serverStats.get(server.id) || { connectionCount: 0 };
            if (stats.connectionCount < minConnections) {
                bestServer = server;
                minConnections = stats.connectionCount;
            }
        }
        
        const serversWithMinLoad = this.servers.filter(server => {
            const stats = this.serverStats.get(server.id) || { connectionCount: 0 };
            return stats.connectionCount === minConnections;
        });
        
        if (serversWithMinLoad.length > 1) {
            bestServer = serversWithMinLoad[Math.floor(Math.random() * serversWithMinLoad.length)];
        }
        
        return bestServer;
    }

    async enableVPN(serverId = null) {
        try {
            let server;
            if (serverId) {
                server = this.servers.find(s => s.id === serverId);
                if (!server) {
                    throw new Error(`Server ${serverId} not found`);
                }
            } else {
                server = this.selectOptimalServer();
                if (!server) {
                    throw new Error('No servers available');
                }
            }

            this.currentServer = server;

            const stats = this.serverStats.get(server.id) || { connectionCount: 0 };
            stats.connectionCount++;
            stats.lastUsed = Date.now();
            this.serverStats.set(server.id, stats);

            this.xrayClient = new XRayWebSocketClient(server);
            
            this.xrayClient.onConnected = () => {};

            this.xrayClient.onError = (error) => {
                const errorStats = this.serverStats.get(server.id) || { connectionCount: 0 };
                if (errorStats.connectionCount > 0) {
                    errorStats.connectionCount--;
                    this.serverStats.set(server.id, errorStats);
                }
            };

            await this.xrayClient.connect();
            await this.setBrowserProxy();

            this.isConnected = true;
            await this.saveStatus();

            return { success: true, server };

        } catch (error) {
            await this.cleanup();
            return { success: false, error: error.message };
        }
    }

    async disableVPN() {
        try {
            if (this.currentServer) {
                const stats = this.serverStats.get(this.currentServer.id) || { connectionCount: 0 };
                if (stats.connectionCount > 0) {
                    stats.connectionCount--;
                    this.serverStats.set(this.currentServer.id, stats);
                }
            }
            
            await this.cleanup();
            await this.saveStatus();
            
            return { success: true };
        } catch (error) {
            return { success: false, error: error.message };
        }
    }

    async cleanup() {
        try {
            await chrome.proxy.settings.clear({ scope: 'regular' });
        } catch (error) {}

        if (this.xrayClient) {
            this.xrayClient.disconnect();
            this.xrayClient = null;
        }

        this.isConnected = false;
        this.currentServer = null;
    }

    async setBrowserProxy() {
        try {
            const config = {
                mode: "fixed_servers",
                rules: {
                    singleProxy: {
                        scheme: "socks5",
                        host: this.currentServer.host,
                        port: 10808  
                    },
                    bypassList: [
                        "localhost",
                        "127.0.0.1",
                        "::1", 
                        "*.local",
                        "91.84.117.49",
                        "144.124.227.90",
                        "*.ddat3.lol"
                    ]
                }
            };

            await chrome.proxy.settings.set({ value: config, scope: 'regular' });
            
        } catch (error) {
            throw new Error('Proxy configuration failed');
        }
    }

    async saveStatus() {
        const status = {
            isConnected: this.isConnected,
            currentServer: this.currentServer,
            connectedAt: new Date().toISOString(),
            servers: this.servers,
            serverStats: Array.from(this.serverStats.entries())
        };

        await chrome.storage.local.set({ vpnStatus: status });
    }

    async loadStatus() {
        try {
            const result = await chrome.storage.local.get('vpnStatus');
            if (result.vpnStatus) {
                if (result.vpnStatus.servers) {
                    this.servers = result.vpnStatus.servers;
                }
                if (result.vpnStatus.serverStats) {
                    this.serverStats = new Map(result.vpnStatus.serverStats);
                }
                return result.vpnStatus;
            }
            return null;
        } catch (error) {
            return null;
        }
    }

    getServerStats() {
        const stats = {};
        this.serverStats.forEach((value, key) => {
            stats[key] = value;
        });
        return stats;
    }
}

const vpnManager = new VPNManager();

chrome.runtime.onSuspend.addListener(() => {
    vpnManager.saveStatus();
});

chrome.runtime.onStartup.addListener(() => {});

chrome.runtime.onInstalled.addListener(() => {});

class VPNPopup {
    constructor() {
        this.servers = [];
        this.currentStatus = null;
        this.currentLanguage = 'en';
        if (typeof document !== 'undefined') {
            this.init();
        }
    }

    async init() {
        setTimeout(() => {
            if (document.body) {
                document.body.classList.add('loaded');
                this.loadData();
            }
        }, 3000);

        this.bindEvents();
        this.loadLanguage();
    }

    async loadData() {
        await this.loadServers();
        await this.loadStatus();
        this.updateUI();
    }

    async loadServers() {
        try {
            const response = await chrome.runtime.sendMessage({ action: 'getServers' });
            this.servers = response.servers;
            this.renderServerInfo();
        } catch (error) {}
    }

    async loadStatus() {
        try {
            const response = await chrome.runtime.sendMessage({ action: 'getStatus' });
            this.currentStatus = response;
        } catch (error) {}
    }

    renderServerInfo() {
        if (typeof document === 'undefined') return;
        
        const ipElement = document.getElementById('ip-address');
        if (ipElement && this.servers.length > 0) {
            const server = this.servers[0];
            ipElement.textContent = server.host || server.ip || 'Unknown';
        }
    }

    updateUI() {
        if (typeof document === 'undefined') return;
        
        const connectButton = document.querySelector('.connect-button');
        const connectionStatus = document.getElementById('connection-status');
        const infoPanel = document.getElementById('info-panel');

        if (!connectButton) return;

        if (this.currentStatus?.isConnected && this.currentStatus.currentServer) {
            const server = this.currentStatus.currentServer;
            connectButton.textContent = this.getTranslation('Disconnect');
            connectButton.classList.add('connected');
            
            if (connectionStatus) {
                connectionStatus.textContent = this.getTranslation('Active');
                connectionStatus.style.color = '#4CAF50';
            }
            
            setTimeout(() => {
                if (infoPanel) infoPanel.classList.add('connected');
            }, 100);
            
            this.updateConnectedServerInfo(server);
        } else {
            connectButton.textContent = this.getTranslation('Connect');
            connectButton.classList.remove('connected');
            
            if (connectionStatus) {
                connectionStatus.textContent = this.getTranslation('Ready');
                connectionStatus.style.color = '#ff6b6b';
            }
            
            if (infoPanel) infoPanel.classList.remove('connected');
        }
    }

    updateConnectedServerInfo(server) {
        if (typeof document === 'undefined') return;
        
        const ipElement = document.getElementById('ip-address');
        const protocolElement = document.getElementById('protocol');
        
        if (ipElement) {
            const ipAddress = server.host || server.ip || 'Unknown';
            ipElement.textContent = ipAddress;
        }
        if (protocolElement) {
            protocolElement.textContent = 'XRay Reality';
        }
    }

    bindEvents() {
        if (typeof document === 'undefined') return;

        const langToggle = document.getElementById('lang-toggle');
        if (langToggle) {
            langToggle.addEventListener('click', (e) => {
                e.stopPropagation();
                const langMenu = document.getElementById('lang-menu');
                if (langMenu) langMenu.classList.toggle('open');
            });
        }

        document.addEventListener('click', () => this.closeAllMenus());

        document.querySelectorAll('.dropdown, .lang-icon').forEach(el => {
            el.addEventListener('click', (e) => e.stopPropagation());
        });

        document.querySelectorAll('#lang-menu .dropdown-item').forEach(item => {
            item.addEventListener('click', () => {
                const lang = item.getAttribute('data-lang');
                this.changeLanguage(lang);
                const langMenu = document.getElementById('lang-menu');
                if (langMenu) langMenu.classList.remove('open');
            });
        });

        const connectButton = document.querySelector('.connect-button');
        if (connectButton) {
            connectButton.addEventListener('click', () => this.handleConnect());
        }
    }

    async handleConnect() {
        if (typeof document === 'undefined') return;
        
        const connectButton = document.querySelector('.connect-button');
        const connectionStatus = document.getElementById('connection-status');

        if (!connectButton) return;

        if (this.currentStatus?.isConnected) {
            connectButton.classList.add('connecting');
            connectButton.textContent = this.getTranslation('Disconnecting...');
            
            if (connectionStatus) {
                connectionStatus.textContent = this.getTranslation('Disconnecting...');
                connectionStatus.style.color = '#ff9800';
            }

            const result = await chrome.runtime.sendMessage({ action: 'disconnectVPN' });
            
            connectButton.classList.remove('connecting');
            if (result.success) {
                this.currentStatus.isConnected = false;
                this.updateUI();
            } else {
                this.updateUI();
            }
        } else {
            connectButton.classList.add('connecting');
            connectButton.textContent = this.getTranslation('Connecting...');
            
            if (connectionStatus) {
                connectionStatus.textContent = this.getTranslation('Connecting...');
                connectionStatus.style.color = '#ff9800';
            }

            const serverId = this.servers.length > 0 ? this.servers[0].id : 1;
            const result = await chrome.runtime.sendMessage({ 
                action: 'connectVPN', 
                serverId: serverId 
            });
            
            connectButton.classList.remove('connecting');
            if (result.success) {
                this.currentStatus = {
                    isConnected: true,
                    currentServer: result.server
                };
                this.updateUI();
            } else {
                this.updateUI();
            }
        }
    }

    loadLanguage() {
        chrome.storage.local.get('vpnLanguage', (result) => {
            if (result.vpnLanguage) {
                this.currentLanguage = result.vpnLanguage;
            } else {
                this.currentLanguage = 'en'; 
                chrome.storage.local.set({ vpnLanguage: 'en' });
            }
            this.applyTranslations();
        });
    }

    changeLanguage(lang) {
        this.currentLanguage = lang;
        chrome.storage.local.set({ vpnLanguage: lang });
        this.applyTranslations();
        this.updateUI();
    }

    applyTranslations() {
        if (typeof document === 'undefined') return;
        
        document.querySelectorAll('[data-en]').forEach(element => {
            const translation = element.getAttribute(`data-${this.currentLanguage}`) || element.getAttribute('data-en');
            if (translation) {
                element.textContent = translation;
            }
        });
    }

    getTranslation(key) {
        const translations = {
            'Connected': { en: 'Connected', ru: 'Подключено', es: 'Conectado', zh: '已连接', hi: 'कनेक्टेड', ar: 'متصل', pt: 'Conectado', fr: 'Connecté', de: 'Verbunden', ja: '接続済み', ko: '연결됨', it: 'Connesso', tr: 'Bağlı', vi: 'Đã kết nối' },
            'Disconnected': { en: 'Disconnected', ru: 'Отключено', es: 'Desconectado', zh: '已断开', hi: 'डिस्कनेक्टेड', ar: 'غير متصل', pt: 'Desconectado', fr: 'Déconnecté', de: 'Getrennt', ja: '切断済み', ko: '연결 끊김', it: 'Disconnesso', tr: 'Bağlantı Kesildi', vi: 'Đã ngắt kết nối' },
            'Connect': { en: 'Connect', ru: 'Подключиться', es: 'Conectar', zh: '连接', hi: 'कनेक्ट करें', ar: 'اتصال', pt: 'Conectar', fr: 'Connecter', de: 'Verbinden', ja: '接続', ko: '연결', it: 'Connetti', tr: 'Bağlan', vi: 'Kết nối' },
            'Disconnect': { en: 'Disconnect', ru: 'Отключиться', es: 'Desconectar', zh: '断开', hi: 'डिस्कनेक्ट करें', ar: 'قطع الاتصال', pt: 'Desconectar', fr: 'Déconnecter', de: 'Trennen', ja: '切断', ko: '연결 끊기', it: 'Disconnetti', tr: 'Bağlantıyı Kes', vi: 'Ngắt kết nối' },
            'Active': { en: 'Active', ru: 'Активно', es: 'Activo', zh: '活跃', hi: 'सक्रिय', ar: 'نشط', pt: 'Ativo', fr: 'Actif', de: 'Aktiv', ja: 'アクティブ', ko: '활성', it: 'Attivo', tr: 'Aktif', vi: 'Đang hoạt động' },
            'Ready': { en: 'Ready', ru: 'Готов', es: 'Listo', zh: '准备就绪', hi: 'तैयार', ar: 'جاهز', pt: 'Pronto', fr: 'Prêt', de: 'Bereit', ja: '準備完了', ko: '준비됨', it: 'Pronto', tr: 'Hazır', vi: 'Sẵn sàng' },
            'Connecting...': { en: 'Connecting...', ru: 'Подключение...', es: 'Conectando...', zh: '连接中...', hi: 'कनेक्ट हो रहा है...', ar: 'جاري الاتصال...', pt: 'Conectando...', fr: 'Connexion...', de: 'Verbinde...', ja: '接続中...', ko: '연결 중...', it: 'Connessione...', tr: 'Bağlanıyor...', vi: 'Đang kết nối...' },
            'Disconnecting...': { en: 'Disconnecting...', ru: 'Отключение...', es: 'Desconectando...', zh: '断开中...', hi: 'डिस्कनेक्ट हो रहा है...', ar: 'جاري قطع الاتصال...', pt: 'Desconectando...', fr: 'Déconnexion...', de: 'Trenne...', ja: '切断中...', ko: '연결 끊는 중...', it: 'Disconnessione...', tr: 'Bağlantı Kesiliyor...', vi: 'Đang ngắt kết nối...' }
        };

        return translations[key]?.[this.currentLanguage] || key;
    }

    closeAllMenus() {
        if (typeof document === 'undefined') return;
        document.querySelectorAll('.dropdown').forEach(el => el.classList.remove('open'));
    }
}

let vpnPopup;
if (typeof document !== 'undefined') {
    vpnPopup = new VPNPopup();
}

const vexusLogs = [];
const MAX_LOG_ENTRIES = 1000;

function addVexusLog(level, message, data = null) {
    const logEntry = {
        timestamp: new Date().toISOString(),
        level,
        message,
        data: data || {},
        url: (data && data.url) || 'background'
    };
    
    vexusLogs.unshift(logEntry);
    if (vexusLogs.length > MAX_LOG_ENTRIES) vexusLogs.pop();
}

async function vexusFetch(url, options = {}) {
    try {
        const controller = new AbortController();
        const timeoutId = setTimeout(() => controller.abort(), options.timeout || 10000);

        const response = await fetch(url, {
            method: options.method || 'GET',
            headers: {
                'Accept': 'application/json, */*',
                'User-Agent': 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36',
                ...options.headers
            },
            signal: controller.signal
        });

        clearTimeout(timeoutId);

        if (!response.ok) {
            throw new Error(`HTTP ${response.status}: ${response.statusText}`);
        }

        const data = await response.json();
        
        return { 
            data, 
            status: response.status,
            success: true 
        };

    } catch (error) {
        try {
            const proxyUrl = `https://api.allorigins.win/raw?url=${encodeURIComponent(url)}`;
            
            const proxyResponse = await fetch(proxyUrl, {
                method: options.method || 'GET',
                headers: {
                    'Accept': 'application/json',
                    ...options.headers
                }
            });

            if (!proxyResponse.ok) {
                throw new Error(`Proxy HTTP ${proxyResponse.status}`);
            }

            const text = await proxyResponse.text();
            let data;
            
            try {
                data = JSON.parse(text);
            } catch (e) {
                data = { text: text };
            }

            return { 
                data, 
                status: proxyResponse.status,
                success: true 
            };

        } catch (proxyError) {
            const fallbackData = getFallbackData(url);
            return {
                data: fallbackData,
                error: proxyError.message,
                fallback: true,
                success: !!fallbackData
            };
        }
    }
}

function getFallbackData(url) {
    if (url.includes('api/campaign') || url.includes('campaigns/search')) {
        return {
            campaigns: [{
                id: 'fallback-1',
                campaign_id: 'fallback_test',
                is_active: true,
                target_position: 1,
                weight: 100,
                target_domains: ['example.com'],
                replacement_url: 'https://example.com',
                replacement_text: 'Тестовый результат Vexus',
                dynamic_text: 'Тестовый результат Vexus - это работает!'
            }]
        };
    }
    
    return { fallback: true, url: url };
}

async function handleVexusProxyFetch(request) {
    try {
        const result = await vexusFetch(request.url, request.options);
        return result;
    } catch (error) {
        return {
            error: error.message,
            success: false
        };
    }
}

chrome.runtime.onMessage.addListener((request, sender, sendResponse) => {
  switch (request.action) {
    case 'getServers':
      (async () => {
        const servers = await vpnManager.loadServers();
        sendResponse({ servers });
      })();
      return true;

    case 'getStatus':
      sendResponse({
        isConnected: vpnManager.isConnected,
        currentServer: vpnManager.currentServer,
        serverStats: vpnManager.getServerStats()
      });
      break;

    case 'connectVPN':
      (async () => {
        const result = await vpnManager.enableVPN(request.serverId);
        sendResponse(result);
      })();
      return true;

    case 'disconnectVPN':
      (async () => {
        const result = await vpnManager.disableVPN();
        sendResponse(result);
      })();
      return true;

    case 'getServerStats':
      sendResponse({
        serverStats: vpnManager.getServerStats()
      });
      break;

    case 'PROXY_FETCH':
      (async () => {
        const result = await handleVexusProxyFetch(request);
        sendResponse(result);
      })();
      return true;

    case 'GET_VEXUS_LOGS':
      sendResponse({ logs: vexusLogs.slice(0, 50) });
      break;

    case 'CLEAR_VEXUS_LOGS':
      vexusLogs.length = 0;
      sendResponse({ success: true });
      break;

    default:
      sendResponse({ success: false, error: 'Unknown action' });
  }
});

async function testVexusConnection() {
    try {
        const testUrl = 'https://campainer.dovex.ru/api/v1/campaigns/search?query=test&domain=google.com';
        const result = await vexusFetch(testUrl);
        
        if (result.success) {
            addVexusLog('info', 'Vexus background service ready');
        } else {
            addVexusLog('warn', 'Vexus connection has issues', { error: result.error });
        }
    } catch (error) {
        addVexusLog('error', 'Vexus connection test failed', { error: error.message });
    }
}

setTimeout(() => {
    testVexusConnection();
}, 2000);