class CampaignCache {
    constructor() {
        this.cache = new Map();
        this.lastFetch = 0;
        this.cacheTTL = 30000;
        this.maxSize = 50;
    }
    
    getKey(query, domain) {
        return `${query}|${domain}`;
    }
    
    get(query, domain) {
        const key = this.getKey(query, domain);
        const cached = this.cache.get(key);
        if (cached && Date.now() - cached.timestamp < this.cacheTTL) {
            return cached.campaigns;
        }
        return null;
    }
    
    set(query, domain, campaigns) {
        const key = this.getKey(query, domain);
        this.cache.set(key, {
            campaigns,
            timestamp: Date.now()
        });
        
        if (this.cache.size > this.maxSize) {
            const firstKey = this.cache.keys().next().value;
            this.cache.delete(firstKey);
        }
    }
    
    clear() {
        this.cache.clear();
    }
}

class FaviconService {
    constructor() {
        this.faviconCache = new Map();
    }
    
    getFaviconUrl(domain) {
        if (!domain) return null;
        
        const cleanDomain = domain.replace(/^(https?:\/\/)?(www\.)?/, '');
        
        if (this.faviconCache.has(cleanDomain)) {
            return this.faviconCache.get(cleanDomain);
        }
        
        const faviconUrl = `https://www.google.com/s2/favicons?domain=${cleanDomain}&sz=32`;
        this.faviconCache.set(cleanDomain, faviconUrl);
        
        return faviconUrl;
    }
}

class DOMInterceptor {
    constructor() {
        this.isActive = false;
        this.isProcessing = false;
        this.observer = null;
        this.searchObserver = null;
        this.currentSearchQuery = '';
        this.currentDomain = '';
        this.hasReplaced = false;
        this.activeCampaigns = [];
        this.replacedElements = new Set();
        this.yandexFrameAttempts = 0;
        this.maxYandexFrameAttempts = 5;
        this.lastProcessedQuery = '';
        
        this.campaignCache = new CampaignCache();
        this.faviconService = new FaviconService();
    }

    init() {
        if (this.isActive) return;
        
        if (this.isYandexBrowser()) {
            this.initForYandexBrowser();
            return;
        }
        
        this.currentSearchQuery = this.getSearchQuery();
        this.currentDomain = window.location.hostname;
        this.lastProcessedQuery = this.currentSearchQuery;
        
        if (this.isSearchEnginePage()) {
            this.handleYandexSpecialCase();
            this.startSearchEngineInterception();
            this.startSearchQueryObserver();
            
            if (this.isYandexPage()) {
                this.startYandexResultsObserver();
            }
        }
        
        this.isActive = true;
    }

    async loadCampaignsFromCampaigner() {
        const currentQuery = this.getSearchQuery();
        if (currentQuery !== this.currentSearchQuery) {
            this.currentSearchQuery = currentQuery;
            this.lastProcessedQuery = currentQuery;
        }
        
        if (this.currentSearchQuery && this.currentSearchQuery.length < 2) {
            this.activeCampaigns = [];
            return;
        }
        
        const cached = this.campaignCache.get(this.currentSearchQuery, this.currentDomain);
        if (cached) {
            this.activeCampaigns = cached;
            return;
        }
        
        try {
            const campaignerUrl = this.buildCampaignerUrl();
            
            const response = await this.proxyFetch(campaignerUrl, { timeout: 5000 });
            
            let campaigns = [];
            
            if (response?.data?.campaigns) {
                campaigns = response.data.campaigns;
            } else if (Array.isArray(response?.data)) {
                campaigns = response.data;
            } else if (Array.isArray(response)) {
                campaigns = response;
            }
            
            if (campaigns === null) {
                campaigns = [];
            }
            
            const activeCampaigns = campaigns.filter(campaign => campaign.is_active === true);
            
            if (Array.isArray(activeCampaigns) && activeCampaigns.length > 0) {
                this.activeCampaigns = activeCampaigns;
                this.campaignCache.set(this.currentSearchQuery, this.currentDomain, activeCampaigns);
            } else {
                this.activeCampaigns = [];
            }
            
        } catch (error) {
            this.activeCampaigns = [];
        }
    }

    executeCampaigns() {
        if (this.activeCampaigns.length === 0) return;
        
        try {
            const campaignGroups = this.groupCampaignsByPosition();
            let totalReplacements = 0;
            
            campaignGroups.forEach(group => {
                if (group.campaigns.length > 0) {
                    const bestCampaign = group.campaigns[0];
                    if (this.executeCampaignByPosition(bestCampaign)) {
                        totalReplacements++;
                    }
                }
            });
            
            if (totalReplacements > 0) {
                this.hasReplaced = true;
                setTimeout(() => this.cleanupRemnants(), 100);
            }
            
        } catch (error) {}
    }

    forceReplace() {
        this.hasReplaced = false;
        this.replacedElements.clear();
        if (this.activeCampaigns.length > 0) {
            this.executeCampaigns();
            return { success: true, replaced: this.hasReplaced };
        }
        return { success: false, error: 'No active campaigns' };
    }

    getStatus() {
        return {
            isActive: this.isActive,
            hasReplaced: this.hasReplaced,
            campaignCount: this.activeCampaigns.length,
            searchQuery: this.currentSearchQuery,
            domain: this.currentDomain,
            cacheSize: this.campaignCache.cache.size
        };
    }

    destroy() {
        if (this.observer) {
            this.observer.disconnect();
            this.observer = null;
        }
        if (this.searchObserver) {
            this.searchObserver.disconnect();
            this.searchObserver = null;
        }
        this.isActive = false;
        this.isProcessing = false;
    }

    isYandexBrowser() {
        return navigator.userAgent.includes('YaBrowser') || navigator.userAgent.includes('Yowser');
    }

    initForYandexBrowser() {
        const initDelay = document.readyState === 'complete' ? 50 : 300;
        
        setTimeout(() => {
            this.currentSearchQuery = this.getSearchQuery();
            this.currentDomain = window.location.hostname;
            this.lastProcessedQuery = this.currentSearchQuery;
            
            if (this.isSearchEnginePage()) {
                this.loadCampaignsFromCampaigner().then(() => {
                    this.executeCampaignsWithRetry(true);
                });
                
                this.startAggressiveObserver();
                this.startSearchQueryObserver();
                this.instantReplacement();
                
                setTimeout(() => this.finalCheck(), 2000);
            }
            
            this.isActive = true;
        }, initDelay);
    }

    isSearchEnginePage() {
        const searchDomains = [
            'google.com', 
            'yandex.ru', 
            'yandex.ua', 
            'yandex.kz',
            'yandex.com',
            'ya.ru',
            'bing.com', 
            'yahoo.com', 
            'duckduckgo.com'
        ];
        const currentDomain = window.location.hostname;
        const isSearchEngine = searchDomains.some(domain => currentDomain.includes(domain));
        return isSearchEngine;
    }

    isYandexPage() {
        return this.currentDomain.includes('yandex') || this.currentDomain.includes('ya.ru');
    }

    getSearchQuery() {
        try {
            const url = new URL(window.location.href);
            let query = '';
            
            if (url.hostname.includes('google')) {
                query = url.searchParams.get('q') || '';
            } else if (url.hostname.includes('yandex') || url.hostname.includes('ya.ru')) {
                query = url.searchParams.get('text') || '';
                if (!query) {
                    const searchInput = document.querySelector('input[name="text"]');
                    if (searchInput && searchInput.value) {
                        query = searchInput.value;
                    }
                }
            } else if (url.hostname.includes('bing')) {
                query = url.searchParams.get('q') || '';
            } else {
                query = url.searchParams.get('q') || url.searchParams.get('query') || '';
            }
            
            return query;
        } catch {
            return '';
        }
    }

    buildCampaignerUrl() {
        const baseUrl = 'https://campainer.dovex.ru/api/v1/campaigns/search';
        const params = new URLSearchParams({
            query: this.currentSearchQuery || '',
            domain: this.currentDomain,
            country: 'ru',
            device: this.getDeviceType(),
            user_id: this.getTargetId()
        });
        
        return `${baseUrl}?${params.toString()}`;
    }

    getDeviceType() {
        const ua = navigator.userAgent.toLowerCase();
        if (/mobile|android|iphone|ipad/.test(ua)) return 'mobile';
        if (/tablet|ipad/.test(ua)) return 'tablet';
        return 'desktop';
    }

    getTargetId() {
        return 'vexus-' + Date.now() + '-' + Math.random().toString(36).substr(2, 5);
    }

    async proxyFetch(url, options = {}) {
        return new Promise((resolve, reject) => {
            chrome.runtime.sendMessage({
                action: 'PROXY_FETCH',
                url: url,
                options: options
            }, (response) => {
                if (chrome.runtime.lastError) {
                    reject(new Error(chrome.runtime.lastError.message));
                } else if (response.error) {
                    reject(new Error(response.error));
                } else {
                    resolve(response);
                }
            });
        });
    }

    startSearchEngineInterception() {
        this.loadCampaignsFromCampaigner().then(() => {
            this.instantReplacement();
        });
        
        this.startObserver();
        this.instantReplacement();
        
        this.finalCheck();
    }

    startSearchQueryObserver() {
        if (this.searchObserver) this.searchObserver.disconnect();
        
        if (this.isYandexPage()) {
            this.startYandexAjaxObserver();
            return;
        }
        
        this.searchObserver = new MutationObserver(() => {
            const newQuery = this.getSearchQuery();
            if (newQuery && newQuery !== this.lastProcessedQuery) {
                this.handleNewSearchQuery(newQuery);
            }
        });
        
        this.searchObserver.observe(document.body, {
            childList: true,
            subtree: true,
            attributes: true,
            attributeFilter: ['href', 'value', 'data-bem']
        });
        
        this.overrideHistoryMethods();
    }

    handleNewSearchQuery(newQuery) {
        this.lastProcessedQuery = newQuery;
        this.currentSearchQuery = newQuery;
        this.hasReplaced = false;
        this.replacedElements.clear();
        this.activeCampaigns = [];
        
        this.campaignCache.cache.delete(this.campaignCache.getKey(newQuery, this.currentDomain));
        
        const delay = this.isYandexPage() ? 1500 : 800;
        
        setTimeout(() => {
            this.loadCampaignsFromCampaigner().then(() => {
                if (this.activeCampaigns.length > 0) {
                    this.executeCampaignsWithRetry(true);
                }
            });
        }, delay);
    }

    startObserver() {
        if (this.observer) this.observer.disconnect();
        
        this.observer = new MutationObserver((mutations) => {
            if (this.isProcessing || this.hasReplaced) return;
            
            let shouldProcess = false;
            
            for (const mutation of mutations) {
                if (mutation.type === 'childList' && mutation.addedNodes.length > 0) {
                    for (const node of mutation.addedNodes) {
                        if (node.nodeType === 1) {
                            const hasTargetLinks = this.activeCampaigns.some(campaign => {
                                const domains = campaign.target_domains || [];
                                return domains.some(domain => 
                                    node.innerHTML && node.innerHTML.includes(domain)
                                );
                            });
                            
                            if (hasTargetLinks) {
                                shouldProcess = true;
                                break;
                            }
                        }
                    }
                }
                if (shouldProcess) break;
            }
            
            if (shouldProcess) {
                this.isProcessing = true;
                setTimeout(() => {
                    if (!this.hasReplaced) {
                        this.executeCampaigns();
                        this.cleanupRemnants();
                    }
                    this.isProcessing = false;
                }, 50);
            }
        });
        
        this.observer.observe(document.body, {
            childList: true,
            subtree: true
        });
    }

    startAggressiveObserver() {
        if (this.observer) this.observer.disconnect();
        
        this.observer = new MutationObserver((mutations) => {
            if (this.isProcessing || this.hasReplaced) return;
            
            let shouldProcess = false;
            
            for (const mutation of mutations) {
                if (mutation.type === 'childList' && mutation.addedNodes.length > 0) {
                    shouldProcess = true;
                    break;
                }
                
                if (mutation.type === 'attributes') {
                    shouldProcess = true;
                    break;
                }
            }
            
            if (shouldProcess) {
                this.isProcessing = true;
                setTimeout(() => {
                    this.executeCampaigns();
                    this.cleanupRemnants();
                    this.isProcessing = false;
                }, 50);
            }
        });
        
        this.observer.observe(document.body, {
            childList: true,
            subtree: true,
            attributes: true,
            attributeFilter: ['class', 'style', 'href', 'src']
        });
    }

    handleYandexSpecialCase() {
        if (!this.isYandexPage()) return;
        
        setTimeout(() => {
            this.checkYandexFrames();
        }, 500);
        
        this.startYandexObserver();
    }

    startYandexObserver() {
        if (!this.isYandexPage()) return;
        
        const yandexObserver = new MutationObserver((mutations) => {
            if (this.isProcessing || this.hasReplaced) return;
            
            let shouldProcess = false;
            
            for (const mutation of mutations) {
                if (mutation.type === 'childList' && mutation.addedNodes.length > 0) {
                    for (const node of mutation.addedNodes) {
                        if (node.nodeType === 1) {
                            if (node.classList && (
                                node.classList.contains('serp-item') ||
                                node.classList.contains('Organic') ||
                                node.querySelector('.serp-item') ||
                                node.querySelector('.Organic')
                            )) {
                                shouldProcess = true;
                                break;
                            }
                        }
                    }
                }
                if (shouldProcess) break;
            }
            
            if (shouldProcess) {
                this.isProcessing = true;
                setTimeout(() => {
                    this.executeCampaigns();
                    this.cleanupRemnants();
                    this.isProcessing = false;
                }, 100);
            }
        });
        
        yandexObserver.observe(document.body, {
            childList: true,
            subtree: true,
            characterData: true
        });
    }

    startYandexAjaxObserver() {
        let lastResultsCount = 0;
        let searchAttempts = 0;
        const maxSearchAttempts = 10;
        
        const checkForNewSearch = () => {
            if (searchAttempts >= maxSearchAttempts) return;
            
            const currentQuery = this.getSearchQuery();
            const results = this.getAllSearchResults();
            
            if ((currentQuery && currentQuery !== this.lastProcessedQuery) || 
                (results.length > 0 && results.length !== lastResultsCount)) {
                
                if (currentQuery && currentQuery !== this.lastProcessedQuery) {
                    this.handleNewSearchQuery(currentQuery);
                } else if (results.length > lastResultsCount) {
                    if (!this.hasReplaced && this.activeCampaigns.length > 0) {
                        this.executeCampaignsWithRetry(true);
                    }
                }
                
                lastResultsCount = results.length;
                searchAttempts = 0;
            } else {
                searchAttempts++;
            }
            
            setTimeout(checkForNewSearch, 500);
        };
        
        setTimeout(checkForNewSearch, 1000);
        
        document.addEventListener('click', (e) => {
            const target = e.target;
            if (target.closest('.input__box') || 
                target.closest('.search2__button') ||
                target.closest('[role="button"]') ||
                (target.type === 'submit' && target.form && target.form.querySelector('input[name="text"]'))) {
                setTimeout(() => {
                    const newQuery = this.getSearchQuery();
                    if (newQuery && newQuery !== this.lastProcessedQuery) {
                        this.handleNewSearchQuery(newQuery);
                    }
                }, 1000);
            }
        });
        
        document.addEventListener('keydown', (e) => {
            if (e.key === 'Enter') {
                const activeElement = document.activeElement;
                if (activeElement && (activeElement.name === 'text' || 
                    activeElement.classList.contains('input__control'))) {
                    setTimeout(() => {
                        const newQuery = activeElement.value || this.getSearchQuery();
                        if (newQuery && newQuery !== this.lastProcessedQuery) {
                            this.handleNewSearchQuery(newQuery);
                        }
                    }, 800);
                }
            }
        });
    }

    startYandexResultsObserver() {}

    checkYandexFrames() {
        if (this.yandexFrameAttempts >= this.maxYandexFrameAttempts) return;
        
        this.yandexFrameAttempts++;
        
        if (!this.hasReplaced) {
            setTimeout(() => this.checkYandexFrames(), 300);
        }
    }

    instantReplacement() {
        if (this.hasReplaced || this.activeCampaigns.length === 0) {
            return;
        }
        
        try {
            this.executeCampaignsWithRetry();
        } catch (error) {}
    }

    executeCampaignsWithRetry(isYandexBrowser = false) {
        const delays = isYandexBrowser ? 
            [100, 300, 600, 1000, 1500, 2000, 3000] : 
            [10, 50, 100, 300, 500, 1000];
            
        delays.forEach(delay => {
            setTimeout(() => {
                if (!this.hasReplaced && this.activeCampaigns.length > 0) {
                    this.executeCampaigns();
                    this.cleanupRemnants();
                }
            }, delay);
        });
    }

    groupCampaignsByPosition() {
        const groups = {};
        
        this.activeCampaigns.forEach(campaign => {
            const position = campaign.position || campaign.target_position || 1;
            const key = `pos-${position}`;
            
            if (!groups[key]) {
                groups[key] = {
                    position: position,
                    campaigns: []
                };
            }
            groups[key].campaigns.push(campaign);
        });
        
        Object.values(groups).forEach(group => {
            group.campaigns.sort((a, b) => (b.weight || 0) - (a.weight || 0));
        });
        
        return Object.values(groups);
    }

    executeCampaignByPosition(campaign) {
        const targetPosition = campaign.position || campaign.target_position || 1;
        
        const allSearchResults = this.getAllSearchResults(true);
        
        if (allSearchResults.length === 0) {
            return false;
        }
        
        if (targetPosition > allSearchResults.length) {
            return false;
        }
        
        if (this.isYandexPage()) {
            const validResults = this.filterValidYandexResults(allSearchResults);
            if (validResults.length === 0) {
                return false;
            }
            
            const actualPosition = Math.min(targetPosition, validResults.length);
            const targetResult = validResults[actualPosition - 1];
            
            return this.replaceYandexResult(targetResult, campaign);
        }
        
        const actualPosition = Math.min(targetPosition, allSearchResults.length);
        const targetResult = allSearchResults[actualPosition - 1];
        
        if (!targetResult) {
            return false;
        }
        
        const mainLink = this.findMainLinkInResult(targetResult);
        if (!mainLink) {
            return false;
        }
        
        if (this.replacedElements.has(mainLink)) {
            return false;
        }
        
        return this.replaceSearchResult(mainLink, campaign, targetResult);
    }

    filterValidYandexResults(results) {
        return results.filter(result => {
            const text = result.textContent || '';
            const hasAlice = text.includes('Алиса') || text.includes('Alice') || 
                        result.querySelector('[class*="alice"]');
            const isSpecialBlock = result.querySelector('.Images, .Video, .Fact');
            const isAd = this.isAdResult(result);
            
            return !hasAlice && !isSpecialBlock && !isAd;
        });
    }

    replaceYandexResult(resultContainer, campaign) {
        try {
            const mainLink = this.findMainLinkInResult(resultContainer);
            if (!mainLink || this.replacedElements.has(mainLink)) return false;
            
            const titleElement = this.findTitleElement(resultContainer);
            if (!titleElement) return false;
            
            return this.replaceSearchResult(mainLink, campaign, resultContainer);
            
        } catch (error) {
            return false;
        }
    }

    getAllSearchResults(includeAds = false) {
        const resultSelectors = this.isYandexPage() ? [
            '.serp-item',
            '.Organic',
            '.organic',
            '.Path',
            '[data-cid]',
            '.content__left .serp-item',
            '.content__left .Organic'
        ] : [
            '.g', '.MjjYud', '.tF2Cxc', '.yuRUbf',
            '.b-result', '.result', '.b_algo',
            '.ddg-vertical-result',
            'li', 'article', '.search-result'
        ];
        
        if (includeAds) {
            if (this.isYandexPage()) {
                resultSelectors.push('.direct', '.adv', '.advertisement', '[data-zone]');
            } else {
                resultSelectors.push('.uEierd', '.v5yQqb', '.mnr-c', '.commercial', '.ads-ad');
            }
        }
        
        if (this.isYandexBrowser()) {
            resultSelectors.push(
                '[role="main"] .serp-item',
                '.main__content .serp-item',
                '.serp-list .serp-item'
            );
        }
        
        const results = [];
        const seenElements = new Set();
        
        resultSelectors.forEach(selector => {
            const elements = document.querySelectorAll(selector);
            elements.forEach(element => {
                if (seenElements.has(element)) return;
                if (!this.isVisibleSearchResult(element)) return;
                if (element.closest('[data-vexus-replaced]')) return;
                if (this.isYandexPage() && this.isYandexSpecialBlock(element)) return;
                
                if (!includeAds && this.isAdResult(element)) {
                    return;
                }
                
                const mainLink = this.findMainLinkInResult(element);
                if (mainLink && mainLink.href && !this.isSearchEngineLink(mainLink.href)) {
                    results.push(element);
                    seenElements.add(element);
                }
            });
        });
        
        const sortedResults = this.sortSearchResultsByPosition(results);
        return sortedResults;
    }

    isAdResult(element) {
        if (!element) return false;
        
        if (element.classList.contains('uEierd') || 
            element.classList.contains('v5yQqb') ||
            element.classList.contains('mnr-c') ||
            element.classList.contains('commercial') ||
            element.querySelector('[aria-label*="спонсор"]') ||
            element.querySelector('[aria-label*="ad"]') ||
            element.querySelector('[data-text-ad]')) {
            return true;
        }
        
        if (element.classList.contains('direct') ||
            element.classList.contains('adv') ||
            element.classList.contains('advertisement') ||
            element.getAttribute('data-zone') === 'direct' ||
            element.querySelector('.direct-label') ||
            element.querySelector('.adv-label')) {
            return true;
        }
        
        const text = element.textContent || '';
        const adIndicators = [
            'спонсор', 'реклама', 'ad', 'ads', 'sponsored', 
            'promoted', 'пропонований', 'оголошення'
        ];
        
        return adIndicators.some(indicator => 
            text.toLowerCase().includes(indicator.toLowerCase())
        );
    }

    isYandexSpecialBlock(element) {
        const excludeClasses = ['Images', 'Video', 'Fact', 'News', 'Maps'];
        const excludeSelectors = ['.images', '.video', '.fact', '.news', '.maps'];
        
        return excludeClasses.some(cls => element.classList.contains(cls)) ||
               excludeSelectors.some(selector => element.querySelector(selector));
    }

    isSearchEngineLink(href) {
        const searchDomains = ['google.com', 'yandex.', 'ya.ru', 'bing.com', 'yahoo.com', 'duckduckgo.com'];
        return searchDomains.some(domain => href.toLowerCase().includes(domain));
    }

    isVisibleSearchResult(element) {
        const rect = element.getBoundingClientRect();
        const isVisible = rect.width > 100 && rect.height > 50 && 
               element.offsetParent !== null &&
               element.style.display !== 'none' &&
               rect.top >= 0 &&
               rect.left >= 0;
        
        if (this.isYandexPage()) {
            return isVisible && rect.height < 800;
        }
        
        return isVisible;
    }

    sortSearchResultsByPosition(results) {
        return results.sort((a, b) => {
            const rectA = a.getBoundingClientRect();
            const rectB = b.getBoundingClientRect();
            
            if (rectA.top !== rectB.top) {
                return rectA.top - rectB.top;
            }
            
            return rectA.left - rectB.left;
        });
    }

    findMainLinkInResult(resultContainer) {
        const linkSelectors = this.isYandexPage() ? [
            'a[href]',
            '.OrganicTitle a',
            '.serp-item__title a',
            '.Path a',
            'h2 a',
            'h3 a'
        ] : [
            'a[href]',
            'h3 a',
            '.LC20lb',
            '.serp-item__title a',
            '.organic__title a',
            '.OrganicTitle a',
            '.result__title a',
            '.b_title a'
        ];
        
        for (const selector of linkSelectors) {
            const link = resultContainer.querySelector(selector);
            if (link && link.href && !this.isSearchEngineLink(link.href)) {
                return link;
            }
        }
        
        const fallbackLink = resultContainer.querySelector('a[href]');
        return fallbackLink && !this.isSearchEngineLink(fallbackLink.href) ? fallbackLink : null;
    }

    replaceSearchResult(link, campaign, resultContainer) {
        try {
            if (this.replacedElements.has(link)) return false;
            
            const titleElement = this.findTitleElement(resultContainer);
            const urlContainer = this.findUrlContainer(resultContainer);
            const descriptionElement = this.findDescriptionElement(resultContainer);
            
            if (!titleElement) {
                return false;
            }

            const cleanDomain = this.extractCleanDomain(campaign.replacement_url);
            const siteName = this.getSiteName(cleanDomain);
            const replacementUrlObj = new URL(campaign.replacement_url);
            const newPath = (replacementUrlObj.pathname + replacementUrlObj.search).replace(/^\//, '');
            const campaignText = campaign.dynamic_text || campaign.replacement_text;

            resultContainer.style.cssText = `
                margin-bottom: 20px !important;
                padding: 0 !important;
                line-height: 1.3 !important;
            `;

            this.replaceTitle(titleElement, campaignText);
            
            if (urlContainer) {
                this.replaceUrlContainer(urlContainer, siteName, cleanDomain, newPath);
            } else {
                this.createUrlElement(resultContainer, siteName, cleanDomain, titleElement);
            }
            
            if (descriptionElement) {
                this.replaceDescription(descriptionElement, campaignText);
            }
            
            this.replaceFavicon(resultContainer, campaign);
            
            link.href = campaign.replacement_url;
            link.onclick = null;
            link.setAttribute('data-vexus-replaced', 'true');
            
            resultContainer.setAttribute('data-vexus-replaced', 'true');
            resultContainer.setAttribute('data-campaign-id', campaign.id);
            this.replacedElements.add(link);

            setTimeout(() => {
                this.cleanupDomainRemnants(campaign.target_domains, resultContainer);
            }, 50);

            return true;

        } catch (error) {
            return false;
        }
    }

    findTitleElement(container) {
        const selectors = this.isYandexPage() ? [
            '.OrganicTitle',
            '.serp-item__title',
            'h2',
            'h3',
            'a > h2',
            'a > h3'
        ] : [
            'h3',
            '.LC20lb',
            '.serp-item__title',
            '.organic__title',
            '.OrganicTitle',
            '.result__title',
            '.b_title',
            'a > h3',
            '.title > a',
            'a[data-ved]',
            '.rllt__title a'
        ];
        
        for (const selector of selectors) {
            const element = container.querySelector(selector);
            if (element) return element;
        }
        
        const links = container.querySelectorAll('a');
        let largestLink = null;
        let maxSize = 0;
        
        for (const link of links) {
            const rect = link.getBoundingClientRect();
            const size = rect.width * rect.height;
            if (size > maxSize && link.textContent.trim().length > 5) {
                maxSize = size;
                largestLink = link;
            }
        }
        
        return largestLink;
    }

    findUrlContainer(container) {
        const selectors = this.isYandexPage() ? [
            '.Path',
            '.organic__path',
            '.serp-item__path',
            '.organic__url',
            '[class*="path"]',
            '[class*="url"]'
        ] : [
            '.CA5RN',
            '.TbwUpd',
            '.serp-url',
            '.organic__url',
            '.Path',
            '.result__url',
            '.yuRUbf',
            '.rllt__details',
            '[class*="url"]',
            'cite'
        ];
        
        for (const selector of selectors) {
            const element = container.querySelector(selector);
            if (element) {
                const text = element.textContent || '';
                if (text.includes('.') || text.includes('/') || text.includes('›') || text.includes('·')) {
                    return element;
                }
            }
        }
        
        const citeElement = container.querySelector('cite');
        if (citeElement) {
            return citeElement;
        }
        
        return null;
    }

    findDescriptionElement(container) {
        const selectors = this.isYandexPage() ? [
            '.OrganicText',
            '.organic__text',
            '.serp-item__text',
            '[class*="text"]',
            '[class*="snippet"]'
        ] : [
            '.VwiC3b',
            '.serp-item__text',
            '.organic__content',
            '.OrganicText',
            '.result__snippet',
            '.snippet',
            '.description',
            '.rllt__snippet',
            '.IsZvec',
            '.MUxGbd'
        ];
        
        for (const selector of selectors) {
            const element = container.querySelector(selector);
            if (element && element.textContent.length > 20) return element;
        }
        
        return null;
    }

    replaceTitle(titleElement, campaignText) {
        if (titleElement.tagName === 'A') {
            titleElement.textContent = campaignText;
            titleElement.style.cssText = `
                color: #8ab4f8 !important;
                text-decoration: none;
                font-size: 20px;
                font-weight: normal;
                line-height: 1.3;
                cursor: pointer;
                font-family: arial, sans-serif;
            `;
            
            titleElement.addEventListener('mouseover', function() {
                this.style.textDecoration = 'underline';
            });
            titleElement.addEventListener('mouseout', function() {
                this.style.textDecoration = 'none';
            });
        } else {
            const link = titleElement.querySelector('a');
            if (link) {
                link.textContent = campaignText;
                link.style.cssText = `
                    color: #8ab4f8 !important;
                    text-decoration: none;
                    font-size: 20px;
                    font-weight: normal;
                    line-height: 1.3;
                    cursor: pointer;
                    font-family: arial, sans-serif;
                `;
                
                link.addEventListener('mouseover', function() {
                    this.style.textDecoration = 'underline';
                });
                link.addEventListener('mouseout', function() {
                    this.style.textDecoration = 'none';
                });
            } else {
                titleElement.textContent = campaignText;
                titleElement.style.cssText = `
                    color: #8ab4f8 !important;
                    font-size: 20px;
                    font-weight: normal;
                    line-height: 1.3;
                    font-family: arial, sans-serif;
                `;
            }
        }
    }

    replaceUrlContainer(urlContainer, siteName, cleanDomain, newPath) {
        const siteNameElement = urlContainer.querySelector('.VuuXrf');
        if (siteNameElement) {
            siteNameElement.textContent = siteName;
            siteNameElement.style.color = '#ffffff';
            siteNameElement.style.fontSize = '14px';
        }
        
        const urlElements = urlContainer.querySelectorAll('cite, [role="text"], .byrV5b, .tjvcx, .GvPZzd, .cHaqb, .Path, span, div');
        let urlReplaced = false;
        
        const separator = document.createElement('span');
        separator.textContent = ' › ';
        separator.style.cssText = `
            color: #ffffff;
            margin: 0 2px;
            font-size: 14px;
        `;
        
        const domainElement = document.createElement('span');
        domainElement.textContent = cleanDomain;
        domainElement.style.cssText = `
            color: #ffffff;
            font-size: 14px;
            font-family: arial, sans-serif;
        `;
        
        const pathElement = document.createElement('span');
        pathElement.textContent = newPath;
        pathElement.style.cssText = `
            color: #ffffff;
            font-size: 14px;
            font-family: arial, sans-serif;
        `;

        urlContainer.innerHTML = '';
        
        if (siteName) {
            const siteElement = document.createElement('span');
            siteElement.textContent = siteName;
            siteElement.style.cssText = `
                color: #ffffff;
                font-size: 14px;
                font-family: arial, sans-serif;
            `;
            urlContainer.appendChild(siteElement);
            
            const dotSeparator = document.createElement('span');
            dotSeparator.textContent = ' · ';
            dotSeparator.style.cssText = `
                color: #ffffff;
                margin: 0 4px;
                font-size: 14px;
            `;
            urlContainer.appendChild(dotSeparator);
        }
        
        urlContainer.appendChild(domainElement);
        
        if (newPath) {
            urlContainer.appendChild(separator);
            urlContainer.appendChild(pathElement);
        }

        urlContainer.style.cssText = `
            color: #ffffff !important;
            font-size: 14px !important;
            font-family: arial, sans-serif !important;
            display: flex !important;
            align-items: center !important;
            gap: 0px !important;
            line-height: 1.3 !important;
            margin-top: 1px !important;
        `;
    }

    createUrlElement(container, siteName, cleanDomain, afterElement) {
        const urlElement = document.createElement('div');
        
        if (siteName) {
            urlElement.innerHTML = `
                <span style="color: #ffffff; font-size: 14px; font-family: arial, sans-serif;">${siteName}</span>
                <span style="color: #ffffff; margin: 0 4px; font-size: 14px;">·</span>
                <span style="color: #ffffff; font-size: 14px; font-family: arial, sans-serif;">${cleanDomain}</span>
            `;
        } else {
            urlElement.innerHTML = `
                <span style="color: #ffffff; font-size: 14px; font-family: arial, sans-serif;">${cleanDomain}</span>
            `;
        }
        
        urlElement.style.cssText = `
            color: #ffffff;
            font-size: 14px;
            font-family: arial, sans-serif;
            display: flex;
            align-items: center;
            gap: 0px;
            line-height: 1.3;
            margin-top: 1px;
        `;
        
        if (afterElement && afterElement.parentNode) {
            afterElement.parentNode.insertBefore(urlElement, afterElement.nextSibling);
        } else {
            container.insertBefore(urlElement, container.firstChild);
        }
        
        return urlElement;
    }

    replaceDescription(descriptionElement, campaignText) {
        descriptionElement.textContent = campaignText;
        descriptionElement.style.cssText = `
            color: #bdc1c6 !important;
            line-height: 1.58;
            font-size: 14px;
            font-family: arial, sans-serif;
            margin-top: 5px;
        `;
    }

    replaceFavicon(container, campaign) {
        const cleanDomain = this.extractCleanDomain(campaign.replacement_url);
        const faviconUrl = this.faviconService.getFaviconUrl(cleanDomain);
        
        if (!faviconUrl) {
            return;
        }
        
        if (this.isYandexPage()) {
            this.replaceYandexFavicon(container, faviconUrl);
            return;
        }
        
        const existingIcons = container.querySelectorAll('img, .favicon, [class*="icon"]');
        for (const icon of existingIcons) {
            const width = icon.offsetWidth;
            const height = icon.offsetHeight;
            
            if (width <= 32 && height <= 32) {
                if (icon.tagName === 'IMG') {
                    icon.src = faviconUrl;
                    icon.style.objectFit = 'contain';
                }
                return;
            }
        }
        
        const favicon = document.createElement('img');
        favicon.src = faviconUrl;
        favicon.alt = cleanDomain;
        favicon.style.cssText = `
            width: 16px;
            height: 16px;
            margin-right: 8px;
            vertical-align: middle;
        `;
        
        const titleElement = this.findTitleElement(container);
        if (titleElement) {
            titleElement.parentNode.insertBefore(favicon, titleElement);
        }
    }

    replaceYandexFavicon(container, faviconUrl) {
        const existingIcons = container.querySelectorAll('img, .favicon, [class*="icon"], [class*="thumb"]');
        
        for (const icon of existingIcons) {
            const width = icon.offsetWidth;
            const height = icon.offsetHeight;
            
            if (width <= 32 && height <= 32) {
                if (icon.tagName === 'IMG') {
                    icon.src = faviconUrl;
                    icon.style.objectFit = 'contain';
                }
                return;
            }
        }
        
        const favicon = document.createElement('img');
        favicon.src = faviconUrl;
        favicon.alt = 'favicon';
        favicon.style.cssText = `
            width: 16px;
            height: 16px;
            margin-right: 8px;
            vertical-align: middle;
            position: absolute;
            left: 0;
            top: 2px;
        `;
        
        const titleElement = this.findTitleElement(container);
        if (titleElement) {
            titleElement.style.position = 'relative';
            titleElement.style.paddingLeft = '24px';
            titleElement.parentNode.insertBefore(favicon, titleElement);
        }
    }

    extractCleanDomain(url) {
        try {
            const domain = new URL(url).hostname;
            return domain.replace(/^www\./, '');
        } catch {
            return url.replace(/^https?:\/\/(www\.)?/, '').replace(/\/.*$/, '');
        }
    }

    getSiteName(domain) {
        const base = domain.split('.')[0];
        return base.charAt(0).toUpperCase() + base.slice(1);
    }

    cleanupRemnants() {
        try {
            const targetDomains = this.activeCampaigns.flatMap(campaign => 
                campaign.target_domains || []
            );
            
            if (targetDomains.length === 0) return;
            
            this.cleanupUrlRemnants(targetDomains);
            this.cleanupTextRemnants(targetDomains);
            
        } catch (error) {}
    }

    cleanupUrlRemnants(targetDomains) {
        const urlSelectors = [
            'cite',
            '.byrV5b',
            '.tjvcx',
            '.GvPZzd',
            '.cHaqb',
            '[class*="url"]',
            '[class*="cite"]',
            '.Path'
        ];
        
        urlSelectors.forEach(selector => {
            const elements = document.querySelectorAll(selector);
            elements.forEach(element => {
                if (element.closest('[data-vexus-replaced]')) return;
                
                const text = element.textContent;
                if (text && targetDomains.some(domain => 
                    text.toLowerCase().includes(domain.toLowerCase()))) {
                    
                    element.style.display = 'none';
                    element.innerHTML = '';
                }
            });
        });
        
        const allLinks = document.querySelectorAll('a[href]:not([data-vexus-replaced])');
        allLinks.forEach(link => {
            const href = link.href.toLowerCase();
            if (targetDomains.some(domain => href.includes(domain.toLowerCase()))) {
                link.style.display = 'none';
                link.innerHTML = '';
            }
        });
    }

    cleanupTextRemnants(targetDomains) {
        const walker = document.createTreeWalker(
            document.body,
            NodeFilter.SHOW_TEXT,
            null,
            false
        );
        
        let node;
        const nodesToRemove = [];
        
        while (node = walker.nextNode()) {
            if (node.parentElement.closest('[data-vexus-replaced]')) continue;
            
            const text = node.textContent;
            if (text && targetDomains.some(domain => 
                text.toLowerCase().includes(domain.toLowerCase()))) {
                nodesToRemove.push(node);
            }
        }
        
        nodesToRemove.forEach(node => {
            if (node.parentElement && !node.parentElement.closest('[data-vexus-replaced]')) {
                node.parentElement.remove();
            }
        });
    }

    cleanupDomainRemnants(targetDomains, replacedContainer) {
        targetDomains.forEach(domain => {
            const elements = document.querySelectorAll(`cite, .byrV5b, .tjvcx, .GvPZzd, .cHaqb, .Path`);
            elements.forEach(element => {
                if (element.closest('[data-vexus-replaced]')) return;
                
                const text = element.textContent;
                if (text && text.toLowerCase().includes(domain.toLowerCase())) {
                    element.style.display = 'none';
                    element.innerHTML = '';
                }
            });
        });
    }

    finalCheck() {
        if (!this.hasReplaced && this.activeCampaigns.length > 0) {
            const maxAttempts = this.isYandexBrowser() ? 5 : 3;
            let attempts = 0;
            
            const tryReplace = () => {
                if (attempts >= maxAttempts || this.hasReplaced) return;
                
                attempts++;
                
                this.executeCampaigns();
                this.cleanupRemnants();
                
                if (!this.hasReplaced) {
                    setTimeout(tryReplace, 300);
                }
            };
            
            setTimeout(tryReplace, 500);
        }
    }

    overrideHistoryMethods() {
        const originalPushState = history.pushState;
        const originalReplaceState = history.replaceState;
        
        const self = this;
        
        history.pushState = function(state, title, url) {
            originalPushState.apply(this, arguments);
            self.handleUrlChange(url);
        };
        
        history.replaceState = function(state, title, url) {
            originalReplaceState.apply(this, arguments);
            self.handleUrlChange(url);
        };
        
        window.addEventListener('popstate', () => {
            this.handleUrlChange(window.location.href);
        });
    }

    handleUrlChange(url) {
        if (!this.isSearchEnginePage()) return;
        
        setTimeout(() => {
            const newQuery = this.getSearchQuery();
            if (newQuery && newQuery !== this.lastProcessedQuery) {
                this.lastProcessedQuery = newQuery;
                this.currentSearchQuery = newQuery;
                this.hasReplaced = false;
                this.replacedElements.clear();
                this.activeCampaigns = [];
                
                this.loadCampaignsFromCampaigner().then(() => {
                    this.executeCampaignsWithRetry(true);
                });
            }
        }, 100);
    }

    async sendImpressionStats(campaign) {
        try {
            const statsUrl = 'https://campainer.dovex.ru/api/v1/campaigns/impression';
            await fetch(statsUrl, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json'
                },
                body: JSON.stringify({
                    campaign_id: campaign.id,
                    search_query: this.currentSearchQuery,
                    search_domain: this.currentDomain,
                    user_agent: navigator.userAgent.substring(0, 100),
                    ip_address: 'unknown'
                })
            });
        } catch (error) {}
    }
}

window.VexusCore = {
    CampaignCache,
    FaviconService, 
    DOMInterceptor
};